package model;

import java.util.HashMap;
import java.util.Set;
import java.util.TreeSet;

import model.attractioncounters.Attraction;

/**
 * This class can manage the whole set of actions necessary to run properly a
 * Luna Park; It does not only allow the addition of new actractions in the
 * system (each attraction is univocally identified by its ID), but performs the
 * check on each attraction when people try to enter that with their ticket
 * 
 * @author simone
 * @since 1.0
 * @version 1.0
 * @see Pass, {@link AttractionPassChecker}
 *
 */
public class CentralSystem {

	private TreeSet<String> attractionsSet = new TreeSet<String>();
	private HashMap<String, Pass> passes = new HashMap<String, Pass>();
	private int lastID = 0;

	/**
	 * Registers a new attraction in the system
	 * 
	 * @param ID
	 *            : a string the attraction is identified by
	 */
	public void addAttraction(String ID) {
		attractionsSet.add(ID);
	}

	/**
	 * Registers a new pass ticket in the system
	 * 
	 * @param pass
	 */
	public void addPass(Pass pass) {
		String ID = nextID();
		pass.setID(ID);
		passes.put(ID, pass);
	}

	/**
	 * Returns the whole set of attractions registered in this system
	 * 
	 * @return
	 */
	public Set<String> getAttractions() {
		return attractionsSet;
	}

	/**
	 * This method notifies the system that a pass has been used to enter an
	 * attraction
	 * 
	 * @param attractionID
	 *            : a string the attraction is identified by
	 * @param passID
	 *            : a string the pass is identified by
	 */
	public void usedPassAt(String attractionID, String passID) {
		passes.get(passID).usedAt(attractionID);
	}

	/**
	 * Returns the number of remaining entries of a pass ticket for a certain
	 * attraction
	 * 
	 * @param attractionID
	 *            : a string the attraction is identified by
	 * @param passID
	 *            : a string the pass is identified by
	 * @return: the number of remaining entries in that pass for that attraction
	 */
	public int getEntriesLeftAt(String attractionID, String passID) {
		return passes.get(passID).getEntriesLeftAt(attractionID);
	}

	private String nextID() {
		lastID++;
		return lastID + "";
	}

}
